<?php
require_once 'config.php';
require_once 'functions.php';

check_login();
check_role(['administrator']);

$success_message = '';
$error_message = '';

// Ambil daftar prodi untuk dropdown
$prodi_list = [];
$stmt_prodi = $conn->prepare("SELECT id, nama_prodi FROM prodi ORDER BY nama_prodi");
$stmt_prodi->execute();
$result_prodi = $stmt_prodi->get_result();
while ($row = $result_prodi->fetch_assoc()) {
    $prodi_list[] = $row;
}
$stmt_prodi->close();


// --- Logika CRUD Kaprodi ---

// Tambah Kaprodi
if (isset($_POST['add_kaprodi'])) {
    $nama_lengkap = trim($_POST['nama_lengkap']);
    $username = trim($_POST['username']);
    $password = trim($_POST['password']);
    $prodi_id = (int)$_POST['prodi_id'];

    if (empty($nama_lengkap) || empty($username) || empty($password) || empty($prodi_id)) {
        $error_message = "Semua kolom harus diisi.";
    } elseif (strlen($password) < 6) {
        $error_message = "Password minimal 6 karakter.";
    } else {
        // Cek username sudah ada atau belum
        $stmt_check_username = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $stmt_check_username->bind_param("s", $username);
        $stmt_check_username->execute();
        $result_check_username = $stmt_check_username->get_result();

        // Cek prodi sudah memiliki kaprodi atau belum
        $stmt_check_prodi = $conn->prepare("SELECT id FROM users WHERE prodi_id = ? AND role = 'kaprodi'");
        $stmt_check_prodi->bind_param("i", $prodi_id);
        $stmt_check_prodi->execute();
        $result_check_prodi = $stmt_check_prodi->get_result();

        if ($result_check_username->num_rows > 0) {
            $error_message = "Username sudah terdaftar.";
        } elseif ($result_check_prodi->num_rows > 0) {
            $error_message = "Prodi ini sudah memiliki Kaprodi. Harap pilih prodi lain atau edit Kaprodi yang sudah ada.";
        } else {
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            $stmt = $conn->prepare("INSERT INTO users (nama_lengkap, username, password, role, prodi_id) VALUES (?, ?, ?, 'kaprodi', ?)");
            $stmt->bind_param("sssi", $nama_lengkap, $username, $hashed_password, $prodi_id);

            if ($stmt->execute()) {
                $success_message = "Kaprodi berhasil ditambahkan.";
                // Clear form
                $_POST = array();
            } else {
                $error_message = "Gagal menambahkan Kaprodi: " . $stmt->error;
            }
            $stmt->close();
        }
        $stmt_check_username->close();
        $stmt_check_prodi->close();
    }
}

// Edit Kaprodi
if (isset($_POST['edit_kaprodi'])) {
    $id_kaprodi = (int)$_POST['id_kaprodi'];
    $nama_lengkap = trim($_POST['nama_lengkap']);
    $username = trim($_POST['username']);
    $password = trim($_POST['password']); // Bisa kosong jika tidak ingin ganti password
    $prodi_id = (int)$_POST['prodi_id'];

    if (empty($nama_lengkap) || empty($username) || empty($prodi_id)) {
        $error_message = "Semua kolom (kecuali password) harus diisi.";
    } elseif (!empty($password) && strlen($password) < 6) {
        $error_message = "Password minimal 6 karakter jika ingin diganti.";
    } else {
        // Cek username sudah ada atau belum untuk user lain (kecuali user yang sedang diedit)
        $stmt_check_username_unique = $conn->prepare("SELECT id FROM users WHERE username = ? AND id != ?");
        $stmt_check_username_unique->bind_param("si", $username, $id_kaprodi);
        $stmt_check_username_unique->execute();
        $result_check_username_unique = $stmt_check_username_unique->get_result();

        // Cek prodi sudah memiliki kaprodi lain atau belum (kecuali kaprodi yang sedang diedit)
        $stmt_check_prodi_unique = $conn->prepare("SELECT id FROM users WHERE prodi_id = ? AND role = 'kaprodi' AND id != ?");
        $stmt_check_prodi_unique->bind_param("ii", $prodi_id, $id_kaprodi);
        $stmt_check_prodi_unique->execute();
        $result_check_prodi_unique = $stmt_check_prodi_unique->get_result();

        if ($result_check_username_unique->num_rows > 0) {
            $error_message = "Username sudah terdaftar untuk pengguna lain.";
        } elseif ($result_check_prodi_unique->num_rows > 0) {
            $error_message = "Prodi ini sudah memiliki Kaprodi lain. Harap pilih prodi lain.";
        } else {
            $sql = "UPDATE users SET nama_lengkap = ?, username = ?, prodi_id = ? ";
            $params = [$nama_lengkap, $username, $prodi_id];
            $types = "ssi";

            if (!empty($password)) {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $sql .= ", password = ? ";
                $params[] = $hashed_password;
                $types .= "s";
            }
            $sql .= "WHERE id = ? AND role = 'kaprodi'";
            $params[] = $id_kaprodi;
            $types .= "i";

            $stmt = $conn->prepare($sql);
            $stmt->bind_param($types, ...$params);

            if ($stmt->execute()) {
                $success_message = "Data Kaprodi berhasil diperbarui.";
            } else {
                $error_message = "Gagal memperbarui data Kaprodi: " . $stmt->error;
            }
            $stmt->close();
        }
        $stmt_check_username_unique->close();
        $stmt_check_prodi_unique->close();
    }
}

// Hapus Kaprodi
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $id_kaprodi = (int)$_GET['id'];

    $stmt_check = $conn->prepare("SELECT id FROM users WHERE id = ? AND role = 'kaprodi'");
    $stmt_check->bind_param("i", $id_kaprodi);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    if ($result_check->num_rows == 0) {
        $error_message = "Kaprodi tidak ditemukan.";
    } else {
        // Sebelum menghapus kaprodi, atur prodi_id di tabel kegiatan menjadi NULL (jika ON DELETE SET NULL)
        // Atau, jika ingin menghapus kegiatan yang dibuat oleh kaprodi ini juga, maka tambahkan DELETE FROM kegiatan WHERE created_by_kaprodi_id = ?
        // Dalam skema yang kita buat, foreign key di `kegiatan` ke `users` adalah ON DELETE CASCADE,
        // sehingga kegiatan yang dibuat oleh kaprodi ini akan otomatis terhapus.
        // Demikian juga prodi_id di tabel users (untuk mahasiswa) akan menjadi NULL.

        $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND role = 'kaprodi'");
        $stmt->bind_param("i", $id_kaprodi);
        if ($stmt->execute()) {
            $success_message = "Kaprodi berhasil dihapus.";
        } else {
            $error_message = "Gagal menghapus Kaprodi: " . $stmt->error;
        }
        $stmt->close();
    }
    redirect('kaprodi.php'); // Redirect untuk menghindari resubmission form
}


// --- Ambil Data Kaprodi ---
$kaprodi_list = [];
$stmt = $conn->prepare("SELECT u.id, u.nama_lengkap, u.username, p.nama_prodi, u.prodi_id FROM users u JOIN prodi p ON u.prodi_id = p.id WHERE u.role = 'kaprodi' ORDER BY p.nama_prodi ASC");
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $kaprodi_list[] = $row;
}
$stmt->close();

// Ambil data kaprodi untuk form edit jika ada parameter 'action=edit'
$kaprodi_to_edit = null;
if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id'])) {
    $id_kaprodi = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT id, nama_lengkap, username, prodi_id FROM users WHERE id = ? AND role = 'kaprodi'");
    $stmt->bind_param("i", $id_kaprodi);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows == 1) {
        $kaprodi_to_edit = $result->fetch_assoc();
    } else {
        $error_message = "Kaprodi tidak ditemukan.";
        redirect('kaprodi.php'); // Redirect jika tidak valid
    }
    $stmt->close();
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Kaprodi - Administrator</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>Manajemen Kaprodi</h1>
        </header>
        <nav>
            <ul>
                <li><a href="dashboard_admin.php">Dashboard</a></li>
                <li><a href="kaprodi.php">Manajemen Kaprodi</a></li>
                <li><a href="change_password.php">Ganti Password</a></li>
                <li><a href="logout.php">Logout</a></li>
            </ul>
        </nav>
        <main>
            <h2>Kelola Data Kaprodi</h2>
            <?php if (!empty($success_message)): ?>
                <div class="alert alert-success"><?php echo $success_message; ?></div>
            <?php endif; ?>
            <?php if (!empty($error_message)): ?>
                <div class="alert alert-danger"><?php echo $error_message; ?></div>
            <?php endif; ?>

            <h3><?php echo $kaprodi_to_edit ? 'Edit Data Kaprodi' : 'Tambah Kaprodi Baru'; ?></h3>
            <form action="kaprodi.php" method="POST">
                <?php if ($kaprodi_to_edit): ?>
                    <input type="hidden" name="id_kaprodi" value="<?php echo htmlspecialchars($kaprodi_to_edit['id']); ?>">
                <?php endif; ?>
                <div class="form-group">
                    <label for="nama_lengkap">Nama Lengkap:</label>
                    <input type="text" id="nama_lengkap" name="nama_lengkap" value="<?php echo htmlspecialchars($kaprodi_to_edit['nama_lengkap'] ?? ($_POST['nama_lengkap'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="username">Username:</label>
                    <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($kaprodi_to_edit['username'] ?? ($_POST['username'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="password">Password <?php echo $kaprodi_to_edit ? '(Kosongkan jika tidak diubah)' : ''; ?>:</label>
                    <input type="password" id="password" name="password" <?php echo $kaprodi_to_edit ? '' : 'required'; ?>>
                </div>
                <div class="form-group">
                    <label for="prodi_id">Program Studi:</label>
                    <select id="prodi_id" name="prodi_id" required>
                        <option value="">Pilih Prodi</option>
                        <?php foreach ($prodi_list as $prodi): ?>
                            <option value="<?php echo $prodi['id']; ?>"
                                <?php
                                if ($kaprodi_to_edit) {
                                    echo ($kaprodi_to_edit['prodi_id'] == $prodi['id']) ? 'selected' : '';
                                } elseif (isset($_POST['prodi_id']) && $_POST['prodi_id'] == $prodi['id']) {
                                    echo 'selected';
                                }
                                ?>
                            >
                                <?php echo htmlspecialchars($prodi['nama_prodi']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <button type="submit" name="<?php echo $kaprodi_to_edit ? 'edit_kaprodi' : 'add_kaprodi'; ?>" class="btn"><?php echo $kaprodi_to_edit ? 'Update Kaprodi' : 'Tambah Kaprodi'; ?></button>
                    <?php if ($kaprodi_to_edit): ?>
                        <a href="kaprodi.php" class="btn btn-warning">Batal Edit</a>
                    <?php endif; ?>
                </div>
            </form>

            <h3 style="margin-top: 40px;">Daftar Kaprodi</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>No.</th>

                            <th>Nama Lengkap</th>
                            <th>Username</th>
                            <th>Program Studi</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($kaprodi_list)): ?>
                            <?php $no = 1; foreach ($kaprodi_list as $kaprodi): ?>
                                <tr>
                                    <td><?php echo $no++; ?></td>
                                    <td><?php echo htmlspecialchars($kaprodi['nama_lengkap']); ?></td>
                                    <td><?php echo htmlspecialchars($kaprodi['username']); ?></td>
                                    <td><?php echo htmlspecialchars($kaprodi['nama_prodi']); ?></td>
                                    <td>
                                        <a href="kaprodi.php?action=edit&id=<?php echo $kaprodi['id']; ?>" class="btn btn-warning btn-sm">Edit</a>
                                        <a href="kaprodi.php?action=delete&id=<?php echo $kaprodi['id']; ?>" class="btn btn-danger btn-sm confirm-delete">Hapus</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="5">Belum ada Kaprodi yang ditambahkan.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
    <script src="assets/js/script.js"></script>
</body>
</html>