<?php
require_once 'config.php';
require_once 'functions.php';

check_login();
check_role(['kaprodi']);

$kaprodi_id = $_SESSION['user_id'];
$kaprodi_prodi_id = $_SESSION['prodi_id'];

$success_message = '';
$error_message = '';

// --- Logika CRUD Mahasiswa ---

// Tambah Mahasiswa
if (isset($_POST['add_mahasiswa'])) {
    $nama_lengkap = trim($_POST['nama_lengkap']);
    $nim = trim($_POST['nim']);
    $no_wa = trim($_POST['no_wa']);
    $tahun_masuk = (int)$_POST['tahun_masuk'];
    $username = trim($_POST['username']);
    $password = trim($_POST['password']);

    if (empty($nama_lengkap) || empty($nim) || empty($no_wa) || empty($tahun_masuk) || empty($username) || empty($password)) {
        $error_message = "Semua kolom harus diisi.";
    } elseif (strlen($password) < 6) {
        $error_message = "Password minimal 6 karakter.";
    } else {
        // Cek NIM dan Username sudah ada atau belum
        $stmt_check = $conn->prepare("SELECT id FROM users WHERE nim = ? OR username = ?");
        $stmt_check->bind_param("ss", $nim, $username);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        if ($result_check->num_rows > 0) {
            $error_message = "NIM atau Username sudah terdaftar.";
        } else {
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            $stmt = $conn->prepare("INSERT INTO users (nama_lengkap, nim, prodi_id, no_wa, tahun_masuk, username, password, role) VALUES (?, ?, ?, ?, ?, ?, ?, 'mahasiswa')");
            $stmt->bind_param("ssiisss", $nama_lengkap, $nim, $kaprodi_prodi_id, $no_wa, $tahun_masuk, $username, $hashed_password);

            if ($stmt->execute()) {
                $success_message = "Mahasiswa berhasil ditambahkan.";
                // Clear form
                $_POST = array();
            } else {
                $error_message = "Gagal menambahkan mahasiswa: " . $stmt->error;
            }
            $stmt->close();
        }
        $stmt_check->close();
    }
}

// Edit Mahasiswa
if (isset($_POST['edit_mahasiswa'])) {
    $id_mahasiswa = (int)$_POST['id_mahasiswa'];
    $nama_lengkap = trim($_POST['nama_lengkap']);
    $nim = trim($_POST['nim']);
    $no_wa = trim($_POST['no_wa']);
    $tahun_masuk = (int)$_POST['tahun_masuk'];
    $username = trim($_POST['username']);
    $password = trim($_POST['password']); // Bisa kosong jika tidak ingin ganti password

    if (empty($nama_lengkap) || empty($nim) || empty($no_wa) || empty($tahun_masuk) || empty($username)) {
        $error_message = "Semua kolom (kecuali password) harus diisi.";
    } elseif (!empty($password) && strlen($password) < 6) {
        $error_message = "Password minimal 6 karakter jika ingin diganti.";
    } else {
        // Pastikan mahasiswa yang diedit berada di prodi kaprodi yang login
        $stmt_check = $conn->prepare("SELECT id FROM users WHERE id = ? AND prodi_id = ? AND role = 'mahasiswa'");
        $stmt_check->bind_param("ii", $id_mahasiswa, $kaprodi_prodi_id);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        if ($result_check->num_rows == 0) {
            $error_message = "Anda tidak memiliki izin untuk mengedit mahasiswa ini.";
        } else {
            // Cek NIM dan Username sudah ada atau belum untuk user lain (kecuali user yang sedang diedit)
            $stmt_check_unique = $conn->prepare("SELECT id FROM users WHERE (nim = ? OR username = ?) AND id != ?");
            $stmt_check_unique->bind_param("ssi", $nim, $username, $id_mahasiswa);
            $stmt_check_unique->execute();
            $result_check_unique = $stmt_check_unique->get_result();
            if ($result_check_unique->num_rows > 0) {
                $error_message = "NIM atau Username sudah terdaftar untuk mahasiswa lain.";
            } else {
                $sql = "UPDATE users SET nama_lengkap = ?, nim = ?, no_wa = ?, tahun_masuk = ?, username = ? ";
                $params = [$nama_lengkap, $nim, $no_wa, $tahun_masuk, $username];
                $types = "sssis";

                if (!empty($password)) {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $sql .= ", password = ? ";
                    $params[] = $hashed_password;
                    $types .= "s";
                }
                $sql .= "WHERE id = ?";
                $params[] = $id_mahasiswa;
                $types .= "i";

                $stmt = $conn->prepare($sql);
                $stmt->bind_param($types, ...$params);

                if ($stmt->execute()) {
                    $success_message = "Data mahasiswa berhasil diperbarui.";
                } else {
                    $error_message = "Gagal memperbarui data mahasiswa: " . $stmt->error;
                }
                $stmt->close();
            }
            $stmt_check_unique->close();
        }
        $stmt_check->close();
    }
}

// Hapus Mahasiswa
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $id_mahasiswa = (int)$_GET['id'];

    // Pastikan mahasiswa yang dihapus berada di prodi kaprodi yang login
    $stmt_check = $conn->prepare("SELECT id FROM users WHERE id = ? AND prodi_id = ? AND role = 'mahasiswa'");
    $stmt_check->bind_param("ii", $id_mahasiswa, $kaprodi_prodi_id);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    if ($result_check->num_rows == 0) {
        $error_message = "Anda tidak memiliki izin untuk menghapus mahasiswa ini.";
    } else {
        // Hapus kehadiran mahasiswa terlebih dahulu karena ada foreign key constraint
        $stmt_delete_kehadiran = $conn->prepare("DELETE FROM kehadiran_kegiatan WHERE mahasiswa_id = ?");
        $stmt_delete_kehadiran->bind_param("i", $id_mahasiswa);
        $stmt_delete_kehadiran->execute();
        $stmt_delete_kehadiran->close();

        $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->bind_param("i", $id_mahasiswa);
        if ($stmt->execute()) {
            $success_message = "Mahasiswa berhasil dihapus.";
        } else {
            $error_message = "Gagal menghapus mahasiswa: " . $stmt->error;
        }
        $stmt->close();
    }
    redirect('mahasiswa.php'); // Redirect untuk menghindari resubmission form
}


// --- Ambil Data Mahasiswa untuk Prodi Kaprodi yang Login ---
$mahasiswa_list = [];
$stmt = $conn->prepare("SELECT id, nama_lengkap, nim, no_wa, tahun_masuk, username FROM users WHERE role = 'mahasiswa' AND prodi_id = ? ORDER BY tahun_masuk DESC, nama_lengkap ASC");
$stmt->bind_param("i", $kaprodi_prodi_id);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $mahasiswa_list[] = $row;
}
$stmt->close();

// Ambil data mahasiswa untuk form edit jika ada parameter 'action=edit'
$mahasiswa_to_edit = null;
if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id'])) {
    $id_mahasiswa = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT id, nama_lengkap, nim, no_wa, tahun_masuk, username FROM users WHERE id = ? AND prodi_id = ? AND role = 'mahasiswa'");
    $stmt->bind_param("ii", $id_mahasiswa, $kaprodi_prodi_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows == 1) {
        $mahasiswa_to_edit = $result->fetch_assoc();
    } else {
        $error_message = "Mahasiswa tidak ditemukan atau Anda tidak memiliki izin untuk mengedit.";
        redirect('mahasiswa.php'); // Redirect jika tidak valid
    }
    $stmt->close();
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Mahasiswa - Kaprodi</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>Manajemen Mahasiswa</h1>
        </header>
        <nav>
            <ul>
                <li><a href="dashboard_kaprodi.php">Dashboard</a></li>
                <li><a href="mahasiswa.php">Manajemen Mahasiswa</a></li>
                <li><a href="kegiatan.php">Manajemen Kegiatan</a></li>
                <li><a href="change_password.php">Ganti Password</a></li>
                <li><a href="logout.php">Logout</a></li>
            </ul>
        </nav>
        <main>
            <h2>Kelola Mahasiswa Program Studi Anda</h2>
            <?php if (!empty($success_message)): ?>
                <div class="alert alert-success"><?php echo $success_message; ?></div>
            <?php endif; ?>
            <?php if (!empty($error_message)): ?>
                <div class="alert alert-danger"><?php echo $error_message; ?></div>
            <?php endif; ?>

            <h3><?php echo $mahasiswa_to_edit ? 'Edit Data Mahasiswa' : 'Tambah Mahasiswa Baru'; ?></h3>
            <form action="mahasiswa.php" method="POST">
                <?php if ($mahasiswa_to_edit): ?>
                    <input type="hidden" name="id_mahasiswa" value="<?php echo htmlspecialchars($mahasiswa_to_edit['id']); ?>">
                <?php endif; ?>
                <div class="form-group">
                    <label for="nama_lengkap">Nama Lengkap:</label>
                    <input type="text" id="nama_lengkap" name="nama_lengkap" value="<?php echo htmlspecialchars($mahasiswa_to_edit['nama_lengkap'] ?? ($_POST['nama_lengkap'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="nim">NIM:</label>
                    <input type="text" id="nim" name="nim" value="<?php echo htmlspecialchars($mahasiswa_to_edit['nim'] ?? ($_POST['nim'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="no_wa">Nomor WhatsApp:</label>
                    <input type="text" id="no_wa" name="no_wa" value="<?php echo htmlspecialchars($mahasiswa_to_edit['no_wa'] ?? ($_POST['no_wa'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="tahun_masuk">Tahun Masuk:</label>
                    <input type="number" id="tahun_masuk" name="tahun_masuk" min="1900" max="<?php echo date('Y'); ?>" value="<?php echo htmlspecialchars($mahasiswa_to_edit['tahun_masuk'] ?? ($_POST['tahun_masuk'] ?? date('Y'))); ?>" required>
                </div>
                <div class="form-group">
                    <label for="username">Username (NIM):</label>
                    <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($mahasiswa_to_edit['username'] ?? ($_POST['username'] ?? '')); ?>" required>
                </div>
                <div class="form-group">
                    <label for="password">Password <?php echo $mahasiswa_to_edit ? '(Kosongkan jika tidak diubah)' : ''; ?>:</label>
                    <input type="password" id="password" name="password" <?php echo $mahasiswa_to_edit ? '' : 'required'; ?>>
                </div>
                <div class="form-group">
                    <button type="submit" name="<?php echo $mahasiswa_to_edit ? 'edit_mahasiswa' : 'add_mahasiswa'; ?>" class="btn"><?php echo $mahasiswa_to_edit ? 'Update Mahasiswa' : 'Tambah Mahasiswa'; ?></button>
                    <?php if ($mahasiswa_to_edit): ?>
                        <a href="mahasiswa.php" class="btn btn-warning">Batal Edit</a>
                    <?php endif; ?>
                </div>
            </form>

            <h3 style="margin-top: 40px;">Daftar Mahasiswa Prodi Anda</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>No.</th>
                            <th>NIM</th>
                            <th>Nama Lengkap</th>
                            <th>Tahun Masuk</th>
                            <th>No. WA</th>
                            <th>Username</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($mahasiswa_list)): ?>
                            <?php $no = 1; foreach ($mahasiswa_list as $mahasiswa): ?>
                                <tr>
                                    <td><?php echo $no++; ?></td>
                                    <td><?php echo htmlspecialchars($mahasiswa['nim']); ?></td>
                                    <td><?php echo htmlspecialchars($mahasiswa['nama_lengkap']); ?></td>
                                    <td><?php echo htmlspecialchars($mahasiswa['tahun_masuk']); ?></td>
                                    <td><?php echo htmlspecialchars($mahasiswa['no_wa']); ?></td>
                                    <td><?php echo htmlspecialchars($mahasiswa['username']); ?></td>
                                    <td>
                                        <a href="mahasiswa.php?action=edit&id=<?php echo $mahasiswa['id']; ?>" class="btn btn-warning btn-sm">Edit</a>
                                        <a href="mahasiswa.php?action=delete&id=<?php echo $mahasiswa['id']; ?>" class="btn btn-danger btn-sm confirm-delete">Hapus</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7">Belum ada mahasiswa di prodi Anda.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
    <script src="assets/js/script.js"></script>
</body>
</html>